// Filename: AlignImageHelper.jsx
// Photoshop jsx script. Platform tested: Windows, Photoshop CS2
// Written by Marc Nijdam, Okt. 2008
//
// Prerequisites
// 1. Acrobat: A pdf file with unaligned pages, like scanned bookpages.
// 2. Acrobat: TouchUp Object Tool selected
// 3. Photoshop: Create an action which runs this script, assign this to a function key. (e.g. F2)
//
// This script provides a quick way to straighten pages in a pdf document in a Acrobat.
// It could be part of a work flow when processing raw scanned pages in a pdf file.
// 1. Open a pdf document with Acrobat.
// 2. Search for pages with improper alignment. While using the TouchUp Object Tool, right click on the page->Edit Image...
// 3. In Photoshop invoke the AlignImageHelper script (e.g. Press F2). Your page is zoomed to 200% at Full Screen.
// 4. Draw a line for alignment along something that should be vertical or horizontal.
// 5. Run the AlignImageHelper script again (e.g. Press F2). The page is now aligned and saved back to Acrobat.
//
// This script mimics the Photoshop rotate trick:
// You select the Measure Tool, put a line along either an almost vertical or horizontal
// object on a picture, click on menu item Image->Rotate Canvas->Arbitrary...
//
// Due to some limitations the angle from the Measure Tool cannot be retrieved. Therefore this script uses
// the Photoshop line tool instead.
//
// Foolproof
// If there is more than one line at your image this script skips the rotating process and just quits.
//

// enable double clicking from the Macintosh Finder or the Windows Explorer
#target photoshop

// in case we double clicked the file
app.bringToFront();

cTID = function(s) { return app.charIDToTypeID(s); };
sTID = function(s) { return app.stringIDToTypeID(s); };


function findAngle(){// This function mimics the mathematics behind the photoshop 'measure tool'
   var P1 = activeDocument.pathItems[0].subPathItems[0].pathPoints[0].anchor;
   var P2 = activeDocument.pathItems[0].subPathItems[0].pathPoints[3].anchor;

   var dx = Math.abs(P1[0]-P2[0]);
   var dy = Math.abs(P1[1]-P2[1]);
   var c = (180/Math.PI) * Math.atan(dy/dx); // c is an angle between 0 and 90 degrees
   
   if (c > 45){// correct for vertical object alignment
      c = c - 90;
   }
// Angle should not depend on the order
// in which the line is drawn. So we are
// actually only interested in the slope.
   if ( ((P2[0] > P1[0]) && (P2[1] > P1[1])) || ((P2[0] < P1[0]) && (P2[1] < P1[1])) ){
      c=-c;
   }
   return c;
};

doMenuItem = function(item, interactive) {
   var ref = new ActionReference();
   ref.putEnumerated(cTID("Mn  "), cTID("MnIt"), item);

   var desc = new ActionDescriptor();
   desc.putReference(cTID("null"), ref);

   try {
     var mode = (interactive != true ? DialogModes.NO : DialogModes.ALL);
     executeAction(sTID("select"), desc, mode);
   } catch (e) {
     if (!e.message.match("User cancelled")) {
       throw e;
     } else {
       return false;
     }
   }
   return true;
}

function selectTool(tool) {
    var desc = new ActionDescriptor();
        var ref = new ActionReference();
        ref.putClass( sTID(tool) );
    desc.putReference( cTID('null'), ref );
    executeAction( cTID('slct'), desc, DialogModes.NO );
};
   
// debug level: 0-2 (0:disable, 1:break on error, 2:break at beginning)
// $.level = 0;
// debugger; // launch debugger on next line


if (app.documents.length == 0){
   // alert("You must have at least one open document to run this script!");
}
else {
   docRef = app.activeDocument;

   if (docRef.activeLayer.isBackgroundLayer) {// Image is in the background layer, no line has been drawn yet.
      doMenuItem(cTID('ActP')); // Set Zoom to 100%
      doMenuItem(cTID('ZmIn')); // Zoom in on time more. (200 %)
      doMenuItem(sTID('screenModeFullScreen')); // Show full screen mode (Exit this mode with pressing key 'f')
      selectTool('lineTool');// Select the line tool

      
   }
   else if (docRef.artLayers.length == 2) {
      if (docRef.layers[0].kind == LayerKind.SOLIDFILL) {
         var angle = findAngle();

         // remove the line used for measuring angle
         docRef.activeLayer.remove();

         // straighten the image
         docRef.rotateCanvas(angle);

         try {
         activeDocument.save();
         activeDocument.close(SaveOptions.SAVECHANGES);

         } catch (e) {
              alert(e);
         }
      }
   }
}